<?php
/**
 * Clipboard Component
 *
 * @package Open Source Social Network
 * @version redegelde v1
 */

/**
 * Definieer het pad naar de module
 * Define the path to the module
 */
define('__CLIPBOARD__', ossn_route()->com . 'clipboard/');

/**
 * Initialiseer de Clipboard-component
 * Initialize the Clipboard Component
 */
function clipboard_init() {
    
    // Voeg CSS toe met geforceerd nieuwe browser laden
    // Add CSS with forced new browser load
    ossn_extend_view('css/ossn.default', 'clipboard/css/clipboard.css?' . time());

    // Voeg een menu-item toe aan het gebruikersprofiel
    // Add a menu item to the user profile
    ossn_register_menu_item('user_timeline', array(
        'name' => 'clipboard',
        'text' => ossn_print('clipboard:menu'),
        'href' => ossn_site_url('clipboard/' . ossn_loggedin_user()->username),
    ));

    // Registreer een pagina-handler voor de module
    // Register a page handler for the module
    ossn_register_page('clipboard', 'clipboard_page_handler');
}

/**
 * Paginahandler voor Clipboard
 * Page handler for Clipboard
 *
 * @param array $pages Pagina's / Pages
 */
function clipboard_page_handler($pages) {
    if (!ossn_isLoggedin()) {
        ossn_error_page();
        return;
    }

    // Controleer of een gebruikersnaam is opgegeven in de URL
    // Check if a username is provided in the URL
    $username = isset($pages[0]) ? $pages[0] : null;
    $user = $username ? ossn_user_by_username($username) : ossn_loggedin_user();

    // Controleer of $user geldig is
    // Validate the $user object
    if (!$user || !$user instanceof OssnUser) {
        error_log("DEBUG: Ongeldige gebruiker of gebruiker niet gevonden.");
        error_log("DEBUG: Invalid user or user not found.");
        ossn_error_page();
        return;
    }

    // Download-functionaliteit
    // Handle download functionality
    if (isset($pages[1]) && $pages[1] === 'download') {
        clipboard_download_data($user);
        exit;
    }

    // Haal data op
    // Fetch user data
    $data = clipboard_fetch_user_data($user);

    // Standaard overzichtspagina
    // Standard overview page
    $title = ossn_print('clipboard:title', array($user->fullname));
    $content = ossn_plugin_view('clipboard/overview', array(
        'user' => $user,
        'data' => $data,
    ));
    echo ossn_set_page_layout('contents', array(
        'content' => $content,
        'title' => $title,
    ));
}

/**
 * Haal gebruikersdata op
 * Fetch user data
 *
 * @param OssnUser $user Gebruiker / User
 * @return array Gebruikersdata / User data
 */
function clipboard_fetch_user_data($user) {
    // Haal tijdlijnberichten op
    // Fetch timeline posts
    $wall = new OssnWall();
    $user_posts = $wall->GetUserPosts($user, array('page_limit' => 1000));

    // Haal blogs op
    // Fetch blogs
    $blog = new Blog();
    $user_blogs = $blog->getUserBlogs($user);

    return array(
        'posts' => $user_posts,
        'blogs' => $user_blogs,
    );
}

/**
 * Download gebruikersdata
 * Download user data
 *
 * @param OssnUser $user Gebruiker / User
 */
function clipboard_download_data($user) {
    header('Content-Type: text/plain');
    header('Content-Disposition: attachment; filename="user_data.txt"');

    $data = clipboard_fetch_user_data($user);

    // Tijdlijnberichten downloaden
    // Download timeline posts
    echo "Timeline Posts:\n";
    if (!empty($data['posts'])) {
        $comments = new OssnComments();
        foreach ($data['posts'] as $post) {
            $post_data = json_decode($post->description, true);
            $post_text = $post_data['post'] ?? 'No description';
            $post_date = date('d-m-Y H:i', $post->time_created);
            echo "{$post_date}: {$post_text}\n";

            // Controleer op afbeelding bij de post
            // Check for an image in the post
            $photo_url = $post->getPhotoURL();
            if ($photo_url) {
                echo "\tPhoto: {$photo_url}\n";
            }

            // Reacties ophalen en toevoegen
            // Fetch and include comments
            $post_comments = $comments->GetComments($post->guid);
            if (!empty($post_comments)) {
                foreach ($post_comments as $comment) {
                    $comment_text = htmlspecialchars_decode($comment->{'comments:post'} ?? 'No comment text');
                    $comment_date = date('d-m-Y H:i', $comment->time_created);
                    $commenter = ossn_user_by_guid($comment->owner_guid);
                    
                    if ($commenter) {
                        $username = $commenter->username;
                    } else {
                        $username = 'Unknown User';
                        error_log("DEBUG: Reactie geplaatst door onbekende gebruiker met GUID: {$comment->owner_guid}");
                        error_log("DEBUG: Comment posted by unknown user with GUID: {$comment->owner_guid}");
                    }

                    echo "\tReply ({$comment_date}, by {$username}): {$comment_text}\n";

                    // Controleer op afbeelding bij de reactie
                    // Check for an image in the reply
                    $comment_photo = ossn_get_file($comment->guid, 'commentphoto');
                    if ($comment_photo) {
                        $photo_url = $comment_photo->getURL();
                        echo "\t\tPhoto in reply: {$photo_url}\n";
                    }
                }
            } else {
                echo "\tNo replies.\n";
            }
        }
    } else {
        echo "No posts found.\n";
    }

    // Blogs downloaden
    // Download blogs
    echo "\nBlogs:\n";
    if (!empty($data['blogs'])) {
        foreach ($data['blogs'] as $blog_post) {
            $blog_title = htmlspecialchars_decode($blog_post->title);
            $blog_content = htmlspecialchars_decode(strip_tags($blog_post->description));
            $blog_date = date('d-m-Y H:i', $blog_post->time_created);
            $blog_url = $blog_post->profileURL();

            echo "Title: {$blog_title}\n";
            echo "Date: {$blog_date}\n";
            echo "Content:\n{$blog_content}\n";
            echo "URL: {$blog_url}\n";
            echo "-------------------------\n";
        }
    } else {
        echo "No blogs found.\n";
    }
}

// Initialiseer de module
// Initialize the module
ossn_register_callback('ossn', 'init', 'clipboard_init');
