<?php
/**
 * Mobile App Bridge
 *
 * @package   MobileApp
 * @author    Core Team <sales@openteknik.com>
 * @copyright (c) Core Team
 * @license   OPENTEKNIK LLC, COMMERCIAL LICENSE v1.0
 * @link      https://www.openteknik.com/
 */
namespace MobileSDK;
class Push {
		private $settings;
		/**
		 * Initialize the component
		 *
		 * @return void
		 */
		public function __construct() {
				$component      = new \OssnComponents();
				$this->settings = $component->getSettings('MobileSDK');
		}
		/**
		 * To make sure settings are added?
		 *
		 * @return boolean
		 */
		public function isConfigured() {
				if(isset($this->settings) && !empty($this->settings->app_id) && !empty($this->settings->app_secret)) {
						return true;
				}
				return false;
		}
		/**
		 * Connect to API
		 *
		 * @param  array  $config Guzzle config
		 * @param  string $method POST/GET/DELETE
		 *
		 * @return boolean|array
		 */
		private function connect(array $config, $method = 'POST') {
				if(empty($config) || !$this->isConfigured()) {
						return false;
				}
				$endpoint = $config['endpoint'];
				unset($config['endpoint']);
				$config  = json_encode($config);
				$options = array(
						'body'    => $config,
						'headers' => array(
								'Authorization' => "Basic {$this->settings->app_secret}",
								'accept'        => 'application/json',
								'content-type'  => 'application/json',
						),
				);
				require_once MOBILE_SDK . 'vendors/guzzle/vendor/autoload.php';
				$client = new \GuzzleHttp\Client(array(
						'http_errors' => false,
				));
				$response = $client->request($method, "https://onesignal.com/api/v1/{$endpoint}", $options);
				$response = $response->getBody();
				if($response && !empty($response)) {
						return json_decode($response, true);
				}
				return false;
		}
		/**
		 * Delete device by ID
		 *
		 * @param  integer $device_id Device ID to delete
		 *
		 * @return boolean
		 */
		public function deleteDevice($device_id) {
				if(!$this->isConfigured()) {
						return false;
				}
				if(empty($device_id)) {
						return false;
				}
				$message = '';
				$config  = array(
						'endpoint'                 => "players/{$device_id}?app_id={$this->settings->app_id}",
						'include_subscription_ids' => $device_id,
						'contents'                 => array(
								'en' => $message,
						),
				);
				$response = $this->connect($config, 'DELETE');
				if($response && !isset($response['errors'])) {
						return true;
				}
				return false;
		}
		/**
		 * Send push notification to user ID
		 *
		 * @param integer $userguid OSSN user GUID
		 * @param string  $message  Message text
		 *
		 * @return boolean
		 */
		public function sendPush($userguid, $message) {
				if(empty($userguid) || empty($message) || !$this->isConfigured()) {
						return false;
				}
				$bridge       = new \MobileSDK\Bridge();
				$devices_list = $bridge->byUser($userguid);

				if($devices_list) {
						$devices_ids = array();
						foreach ($devices_list as $device) {
								$devices_ids[] = $device->device_id;
						}
						$this->sendNotification($devices_ids, $message);
						return true;
				}
				return false;
		}
		/**
		 * Send push notification to user ID
		 *
		 * @param array  $player_ids Devices IDs
		 * @param string $message  Message text
		 *
		 * @return boolean|string
		 */
		private function sendNotification(array $player_ids, $message) {
				if(empty($player_ids) || empty($message) || !$this->isConfigured()) {
						return false;
				}
				$config = array(
						'endpoint'                 => 'notifications',
						'app_id'                   => $this->settings->app_id,
						'include_subscription_ids' => $player_ids,
						'contents'                 => array(
								'en' => $message,
						),
				);
				$response = $this->connect($config);

				if($response && isset($response['errors'])) {
						if(isset($response['errors']['invalid_player_ids'])) {
								//remove IDS from system
								foreach ($response['errors']['invalid_player_ids'] as $player_id) {
										$bridge = new \MobileSDK\Bridge();
										$bridge->deleteDeviceID($player_id);
										$this->deleteDevice($player_id);
								}
						}
						return false;
				}
				return $response;
		}
}