# Alkebulan AI v1.0 - Complete Documentation

## 📋 Table of Contents
1. [Overview](#overview)
2. [Installation](#installation)
3. [Features](#features)
4. [Architecture](#architecture)
5. [Configuration](#configuration)
6. [API Reference](#api-reference)
7. [Database Schema](#database-schema)
8. [File Structure](#file-structure)
9. [Usage Examples](#usage-examples)
10. [Troubleshooting](#troubleshooting)

---

## Overview

**Alkebulan AI** is a full-featured, production-ready AI component for OSSN (Oxwall Social Network) that provides advanced artificial intelligence capabilities including sentiment analysis, intelligent recommendations, conversational AI, and comprehensive analytics.

### Key Information
- **Version**: 1.0 (Stable)
- **Framework**: OSSN 7.6+
- **PHP**: 7.0+
- **MySQL**: 5.6+
- **Status**: Fully Functional

### Core Capabilities
- 🧠 **Sentiment Analysis** - Emotion and sentiment detection
- 💡 **Recommendations** - Smart content and user suggestions
- 💬 **Chat Assistant** - Conversational AI interface
- 📊 **Analytics** - Detailed usage tracking and reporting
- 🔍 **Entity Recognition** - Extract key entities from text
- 🏷️ **Keyword Extraction** - Identify important keywords
- 📈 **Performance Metrics** - Monitor system performance

---

## Installation

### Prerequisites
- OSSN 7.6 or higher installed
- PHP 7.0+
- MySQL 5.6+
- Web server (Apache, Nginx, etc.)
- Write permissions to OSSN installation directory

### Step-by-Step Installation

1. **Extract Component**
   ```
   Extract the alkebulan.zip to: /your-ossn-path/components/
   ```

2. **Navigate to Admin Panel**
   - Log in to your OSSN admin panel
   - Go to: Admin → Components

3. **Activate Component**
   - Find "Alkebulan AI" in the components list
   - Click "Activate" or "Enable"

4. **Configure Settings**
   - Go to: Admin → Settings → Alkebulan AI
   - Set your API configuration
   - Configure quotas and limits

5. **Verify Installation**
   - Check component status in Admin Panel
   - Navigate to Dashboard → Alkebulan AI
   - Verify all database tables were created

### Database Tables Created
```
1. alkebulan_analysis - Stores analysis results
2. alkebulan_recommendations - Tracks recommendations
3. alkebulan_chat_sessions - Chat session data
4. alkebulan_chat_messages - Chat message storage
5. alkebulan_analytics - Analytics aggregation
6. alkebulan_user_prefs - User preferences
7. alkebulan_usage_log - Feature usage tracking
8. alkebulan_config - Component configuration
```

---

## Features

### 1. Content Analysis
**Intelligent text analysis with multiple dimensions**

- **Sentiment Analysis**: Positive, Neutral, Negative detection
- **Emotion Recognition**: Happy, Sad, Angry, Excited, Confused
- **Entity Recognition**: Extract persons, places, organizations
- **Keyword Extraction**: Identify important keywords with frequency
- **Content Categorization**: Auto-categorize content (Tech, Business, Health, Arts, Sports)
- **Confidence Scoring**: Reliability metrics for all analyses

### 2. Smart Recommendations
**Personalized suggestion engine**

- **Content Recommendations**: Discover relevant posts
- **People Suggestions**: Find users to follow
- **Group Recommendations**: Discover communities
- **Timeline Feed**: Personalized content feed
- **Trending Topics**: Current trending keywords
- **Engagement Tracking**: Track recommendation performance
- **Relevance Scoring**: Detailed relevance metrics

### 3. Chat Assistant
**Conversational AI interface**

- **Natural Language Processing**: Understand user intent
- **Intent Detection**: Classify user intentions
- **Session Management**: Maintain conversation context
- **History Tracking**: Full conversation history
- **Session Summaries**: Auto-generated summaries
- **Smart Suggestions**: Context-aware suggestions
- **Multi-session Support**: Manage multiple conversations

### 4. Advanced Analytics
**Comprehensive performance tracking**

- **Usage Statistics**: Track API calls and token usage
- **Feature Breakdown**: Usage by feature
- **Sentiment Trends**: Track sentiment over time
- **Performance Metrics**: Response times and speed
- **System Health**: Component status monitoring
- **Report Generation**: Create detailed reports
- **Data Export**: Export analytics as JSON
- **Cost Analysis**: Token and API cost tracking

---

## Architecture

### Component Structure
```
alkebulan/
├── ossn_com.xml              # Component manifest
├── ossn_com.php              # Main initialization
├── classes/
│   ├── AIAnalyzer.php        # Content analysis engine
│   ├── AIRecommender.php     # Recommendation system
│   ├── ChatAssistant.php     # Chat AI system
│   └── AIAnalytics.php       # Analytics engine
├── actions/
│   ├── analyze.php           # Analysis endpoint
│   ├── recommend.php         # Recommendations endpoint
│   ├── chat.php              # Chat endpoint
│   └── analytics.php         # Analytics endpoint
├── plugins/default/
│   ├── pages/
│   │   ├── dashboard.php     # Main dashboard
│   │   ├── features.php      # Features showcase
│   │   ├── assistant.php     # Chat interface
│   │   ├── analytics.php     # Analytics dashboard
│   │   └── settings.php      # Settings page
│   ├── js/
│   │   └── alkebulan.js      # Frontend scripts
│   ├── css/
│   │   └── alkebulan.css     # Component styling
│   └── views/                # View templates
├── locale/
│   └── ossn.en.php           # Language strings
└── README.md                 # This file
```

### Class Hierarchy

```
AIAnalyzer
├── analyzeSentiment()
├── categorizeContent()
├── recognizeEntities()
├── extractKeywords()
├── batchAnalyze()
└── getAnalysisHistory()

AIRecommender
├── getContentRecommendations()
├── getPeopleRecommendations()
├── getGroupRecommendations()
├── getTimelineRecommendations()
├── getTrendingContent()
├── trackRecommendationEngagement()
└── getRecommendationMetrics()

ChatAssistant
├── createSession()
├── sendMessage()
├── generateResponse()
├── detectIntent()
├── getConversationHistory()
├── addContext()
├── getSessionSummary()
└── getSuggestions()

AIAnalytics
├── logUsage()
├── getUsageStats()
├── getFeatureUsage()
├── getTrendingTopics()
├── getSentimentTrends()
├── getPerformanceMetrics()
├── generateReport()
├── exportReport()
└── getSystemAnalytics()
```

---

## Configuration

### Admin Settings

Access configuration at: **Admin → Settings → Alkebulan AI**

**Basic Settings**
- Component Status: Enable/Disable
- API Key: Your API configuration
- Max Requests Per Day: Rate limiting (default: 1000)
- Cache Enabled: Toggle result caching
- Analytics Enabled: Toggle usage tracking

### Component Settings File
Edit `ossn_com.php` to modify:

```php
// Maximum daily API requests
$component['settings']['max_requests'] = 1000;

// Enable caching for faster responses
$component['settings']['cache_enabled'] = true;

// Track analytics by default
$component['settings']['analytics_enabled'] = true;
```

### User Preferences
Users can configure via: **Settings → Alkebulan AI**
- Default analysis type
- Recommendation sensitivity
- Chat response style
- Analytics tracking
- Privacy settings

---

## API Reference

### Analysis Endpoint
**URL**: `/action/alkebulan/analyze`

**Request**
```javascript
POST /action/alkebulan/analyze
{
  "action": "sentiment|categorize|entities|keywords",
  "text": "Your text to analyze"
}
```

**Response**
```json
{
  "success": true,
  "data": {
    "sentiment": "positive",
    "confidence": 0.85,
    "emotions": ["happy", "excited"],
    "response_time": 245,
    "tokens_used": 150
  }
}
```

### Recommendations Endpoint
**URL**: `/action/alkebulan/recommend`

**Request**
```javascript
POST /action/alkebulan/recommend
{
  "type": "content|people|groups|timeline|trending",
  "limit": 10
}
```

**Response**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "title": "Post Title",
      "relevance_score": 0.92,
      "reason": "matches your interests"
    }
  ]
}
```

### Chat Endpoint
**URL**: `/action/alkebulan/chat`

**Request**
```javascript
POST /action/alkebulan/chat
{
  "action": "create_session|message|get_history|get_summary",
  "message": "Your message here"
}
```

**Response**
```json
{
  "success": true,
  "data": {
    "session_id": 123,
    "message": "Response from AI",
    "timestamp": 1234567890
  }
}
```

### Analytics Endpoint
**URL**: `/action/alkebulan/analytics`

**Request**
```javascript
POST /action/alkebulan/analytics
{
  "report": "usage|features|sentiment|trending|performance|full_report",
  "period": "today|week|month|quarter|year"
}
```

**Response**
```json
{
  "success": true,
  "data": {
    "period": "month",
    "api_calls": 1234,
    "tokens_used": 45200,
    "response_time_avg": 245
  }
}
```

---

## Database Schema

### alkebulan_analysis
Stores all content analysis results

```sql
CREATE TABLE alkebulan_analysis (
  id BIGINT(20) PRIMARY KEY,
  user_id BIGINT(20) NOT NULL,
  text LONGTEXT,
  sentiment VARCHAR(20),
  confidence DECIMAL(3,2),
  emotions JSON,
  entities JSON,
  keywords JSON,
  category VARCHAR(50),
  response_time INT,
  tokens_used INT,
  created_at TIMESTAMP,
  KEY (user_id),
  KEY (created_at)
)
```

### alkebulan_chat_messages
Stores chat messages and conversations

```sql
CREATE TABLE alkebulan_chat_messages (
  id BIGINT(20) PRIMARY KEY,
  session_id BIGINT(20),
  user_id BIGINT(20),
  message TEXT,
  response TEXT,
  intent VARCHAR(50),
  sentiment VARCHAR(20),
  response_time INT,
  created_at TIMESTAMP,
  KEY (session_id),
  KEY (user_id)
)
```

### alkebulan_usage_log
Tracks feature usage and costs

```sql
CREATE TABLE alkebulan_usage_log (
  id BIGINT(20) PRIMARY KEY,
  user_id BIGINT(20),
  feature VARCHAR(50),
  tokens_used INT,
  response_time INT,
  status VARCHAR(20),
  created_at TIMESTAMP,
  KEY (user_id),
  KEY (feature),
  KEY (created_at)
)
```

---

## File Structure

### Essential Files

**ossn_com.xml** (Component Manifest)
- Defines component metadata
- Lists dependencies
- Registers settings
- Version information

**ossn_com.php** (Main Initialization)
- Handles component activation
- Creates database tables
- Registers hooks and menus
- Page routing
- Admin settings

**classes/AIAnalyzer.php** (1000+ lines)
- Sentiment analysis
- Entity recognition
- Keyword extraction
- Content categorization
- Batch processing

**classes/AIRecommender.php** (900+ lines)
- Content recommendations
- User suggestions
- Community discovery
- Trending identification
- Metrics tracking

**classes/ChatAssistant.php** (1100+ lines)
- Session management
- Message processing
- Intent detection
- Context handling
- Suggestion generation

**classes/AIAnalytics.php** (1400+ lines)
- Usage logging
- Statistics calculation
- Report generation
- Data export
- System monitoring

### Frontend Files

**pages/dashboard.php**
- Main dashboard interface
- Statistics overview
- Quick access buttons
- Recent activity feed

**pages/features.php**
- Feature showcase
- Interactive demos
- Feature descriptions
- Benefits listing

**pages/assistant.php**
- Chat interface
- Session management
- Quick prompts
- Conversation display

**pages/analytics.php**
- Analytics dashboard
- Charts and graphs
- Report generation
- Data export

**pages/settings.php**
- User settings
- Preferences
- Privacy options
- API configuration

---

## Usage Examples

### Using AIAnalyzer Class

```php
require_once 'classes/AIAnalyzer.php';

$analyzer = new AIAnalyzer();

// Analyze sentiment
$result = $analyzer->analyzeSentiment("I love this product!");
echo $result['sentiment'];  // Output: positive

// Extract keywords
$keywords = $analyzer->extractKeywords("This is a sample text");
print_r($keywords);

// Recognize entities
$entities = $analyzer->recognizeEntities("John works at Microsoft");
print_r($entities);
```

### Using AIRecommender Class

```php
require_once 'classes/AIRecommender.php';

$recommender = new AIRecommender();

// Get content recommendations
$recommendations = $recommender->getContentRecommendations(10);

// Get people to follow
$people = $recommender->getPeopleRecommendations(5);

// Get trending content
$trending = $recommender->getTrendingContent(10);
```

### Using ChatAssistant Class

```php
require_once 'classes/ChatAssistant.php';

$chat = new ChatAssistant();

// Create session
$session_id = $chat->createSession(['user_id' => 123]);

// Send message
$response = $chat->sendMessage("Hello!", $session_id);
echo $response['message'];

// Get history
$history = $chat->getConversationHistory($session_id);
```

### Using AIAnalytics Class

```php
require_once 'classes/AIAnalytics.php';

$analytics = new AIAnalytics();

// Log usage
$analytics->logUsage('analysis', 150, 245, 'success');

// Get statistics
$stats = $analytics->getUsageStats('month');

// Generate report
$report = $analytics->generateReport('month');
```

---

## Troubleshooting

### Common Issues

**Issue: Component won't activate**
- Check PHP version (7.0+ required)
- Verify database permissions
- Check error logs in Admin Panel

**Issue: Analysis not working**
- Verify API key configuration
- Check daily quota limits
- Review error messages

**Issue: Chat not responding**
- Check session creation
- Verify user authentication
- Check database connectivity

**Issue: Analytics not showing data**
- Verify tracking is enabled
- Check usage log table
- Review period settings

### Debug Mode

Enable debugging in `ossn_com.php`:

```php
define('ALKEBULAN_DEBUG', true);
```

Check logs in:
```
/your-ossn-path/logs/
```

### Performance Optimization

**Enable Caching**
```php
$component['settings']['cache_enabled'] = true;
```

**Optimize Queries**
- Index frequently searched columns
- Archive old data regularly
- Use pagination for large datasets

**Rate Limiting**
- Set appropriate daily quotas
- Monitor token usage
- Implement request throttling

---

## Support & Updates

**Documentation**: See component dashboard
**Support Forum**: OSSN Community Forums
**Bug Reports**: Admin Panel → System
**Feature Requests**: Component Discussion

---

## License & Credits

**Alkebulan AI v1.0**
Built for OSSN Framework
Fully compatible with OSSN 7.6+

---

## Version History

### v1.0 (Current - Stable)
- ✅ Complete AI analysis engine
- ✅ Recommendation system
- ✅ Chat assistant
- ✅ Analytics dashboard
- ✅ Full admin configuration
- ✅ Mobile responsive
- ✅ Comprehensive documentation

---

## Additional Resources

### Related Components
- Livestream v3.0 - Social media streaming
- User Profiles - Enhanced user management
- Notifications - Advanced notification system

### Developer Guide
See `DEVELOPER.md` for:
- Extending classes
- Custom integrations
- API rate limiting
- Database optimization

### API Documentation
Complete API reference available at:
`/components/alkebulan/docs/api.md`

---

**Last Updated**: <?php echo date('F j, Y'); ?>
**Component Version**: 1.0 (Stable)
**Compatible With**: OSSN 7.6+
