<?php
/**
 * AIAnalyzer Class - Content Analysis Engine
 * Analyzes text, sentiment, entity recognition, and content classification
 */

class AIAnalyzer {
    private $db;
    private $user_id;
    private $cache;
    
    public function __construct($user_id = null) {
        $this->db = ossn_get_database();
        $this->user_id = $user_id;
        $this->cache = [];
    }
    
    /**
     * Analyze content for sentiment
     */
    public function analyzeSentiment($text) {
        $analysis = [
            'text' => $text,
            'type' => 'sentiment',
            'sentiment' => $this->detectSentiment($text),
            'confidence' => $this->calculateConfidence($text),
            'emotions' => $this->extractEmotions($text),
            'keywords' => $this->extractKeywords($text),
            'timestamp' => time()
        ];
        
        return $this->storeAnalysis($analysis);
    }
    
    /**
     * Detect overall sentiment
     */
    private function detectSentiment($text) {
        $positive_words = ['good', 'great', 'awesome', 'excellent', 'love', 'happy', 'wonderful', 'fantastic'];
        $negative_words = ['bad', 'terrible', 'awful', 'hate', 'sad', 'horrible', 'disappointing'];
        
        $text_lower = strtolower($text);
        $pos_count = 0;
        $neg_count = 0;
        
        foreach($positive_words as $word) {
            $pos_count += substr_count($text_lower, $word);
        }
        foreach($negative_words as $word) {
            $neg_count += substr_count($text_lower, $word);
        }
        
        if($pos_count > $neg_count) return 'positive';
        if($neg_count > $pos_count) return 'negative';
        return 'neutral';
    }
    
    /**
     * Calculate confidence score
     */
    private function calculateConfidence($text) {
        $length = strlen($text);
        $words = str_word_count($text);
        
        if($words < 5) return 0.4;
        if($words < 20) return 0.6;
        if($words < 100) return 0.8;
        return 0.95;
    }
    
    /**
     * Extract emotions from text
     */
    private function extractEmotions($text) {
        $emotions = [];
        $emotion_map = [
            'happy' => ['happy', 'glad', 'joyful', 'delighted'],
            'sad' => ['sad', 'upset', 'depressed', 'unhappy'],
            'angry' => ['angry', 'furious', 'mad', 'irritated'],
            'excited' => ['excited', 'thrilled', 'eager', 'enthusiastic'],
            'confused' => ['confused', 'puzzled', 'bewildered', 'uncertain']
        ];
        
        $text_lower = strtolower($text);
        foreach($emotion_map as $emotion => $keywords) {
            foreach($keywords as $keyword) {
                if(strpos($text_lower, $keyword) !== false) {
                    $emotions[$emotion] = true;
                }
            }
        }
        
        return array_keys($emotions);
    }
    
    /**
     * Extract keywords from text
     */
    private function extractKeywords($text, $limit = 10) {
        $words = str_word_count($text, 1);
        $words = array_map('strtolower', $words);
        
        // Remove common stopwords
        $stopwords = ['the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'is', 'was', 'are'];
        $words = array_diff($words, $stopwords);
        
        // Count word frequency
        $frequency = array_count_values($words);
        arsort($frequency);
        
        return array_slice(array_keys($frequency), 0, $limit);
    }
    
    /**
     * Analyze content category
     */
    public function categorizeContent($text) {
        $categories = [];
        
        // Check for various content types
        if($this->containsCategory($text, ['tech', 'code', 'software', 'app', 'develop'])) {
            $categories[] = 'technology';
        }
        if($this->containsCategory($text, ['business', 'sales', 'money', 'finance', 'invest'])) {
            $categories[] = 'business';
        }
        if($this->containsCategory($text, ['health', 'medical', 'doctor', 'fitness', 'exercise'])) {
            $categories[] = 'health';
        }
        if($this->containsCategory($text, ['art', 'music', 'design', 'creative', 'culture'])) {
            $categories[] = 'arts';
        }
        if($this->containsCategory($text, ['sports', 'game', 'play', 'team', 'match'])) {
            $categories[] = 'sports';
        }
        
        return $categories ?: ['general'];
    }
    
    /**
     * Check if text contains category keywords
     */
    private function containsCategory($text, $keywords) {
        $text_lower = strtolower($text);
        foreach($keywords as $keyword) {
            if(strpos($text_lower, $keyword) !== false) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Entity recognition - extract names, places, organizations
     */
    public function recognizeEntities($text) {
        $entities = [
            'persons' => [],
            'places' => [],
            'organizations' => []
        ];
        
        // Simple implementation - in production use NLP library
        $words = explode(' ', $text);
        foreach($words as $word) {
            // Check for capitalized words (potential entities)
            if(ctype_upper($word[0]) && strlen($word) > 2) {
                $entities['persons'][] = $word;
            }
        }
        
        return $entities;
    }
    
    /**
     * Store analysis in database
     */
    private function storeAnalysis($analysis) {
        $data = [
            'user_id' => $this->user_id ?: 0,
            'analysis_type' => $analysis['type'],
            'input_text' => substr($analysis['text'], 0, 1000),
            'output_data' => json_encode($analysis),
            'confidence_score' => $analysis['confidence'] ?? 0.5,
            'status' => 'completed',
            'created' => time()
        ];
        
        $query = $this->db->insert('alkebulan_analysis', $data);
        $analysis['id'] = $query;
        
        return $analysis;
    }
    
    /**
     * Batch analyze multiple texts
     */
    public function batchAnalyze($texts) {
        $results = [];
        foreach($texts as $text) {
            $results[] = $this->analyzeSentiment($text);
        }
        return $results;
    }
    
    /**
     * Get analysis history
     */
    public function getAnalysisHistory($limit = 20) {
        $query = $this->db->select('alkebulan_analysis')
            ->where('user_id', $this->user_id)
            ->order_by('created', 'DESC')
            ->limit($limit)
            ->execute();
        
        return $query->fetch();
    }
}
?>
